<?php
/**
 * Application Status
 *
 * @package     AutomatorWP\Integrations\WP_Job_Manager\Triggers\Application_Status
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Job_Manager_Application_Status extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_job_manager';
    public $trigger = 'wp_job_manager_application_status';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User application to a job of a type gets marked as interviewed, offer extended, hired, rejected or archived', 'automatorwp-pro' ),
            'select_option'     => __( 'User application to a job of a type gets marked as <strong>interviewed, offer extended, hired, rejected or archived</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Term title. %2$s: Application status (interviewed, offer extended, hired, rejected or archived). %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User application to a job of %1$s gets marked as %2$s %3$s time(s)', 'automatorwp-pro' ), '{term}', '{status}', '{times}' ),
            /* translators: %1$s: Term title. */
            'log_label'         => sprintf( __( 'User application to a job of %1$s gets marked as %2$s', 'automatorwp-pro' ), '{term}', '{status}' ),
            'action'            => 'transition_post_status',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'name'              => __( 'Type:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any type', 'automatorwp-pro' ),
                    'taxonomy'          => 'job_listing_type',
                ) ),
                'status' => array(
                    'from' => 'status',
                    'fields' => array(
                        'status' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'interviewed' => __( 'Interviewed', 'automatorwp-pro' ),
                                'offer'       => __( 'Offer extended', 'automatorwp-pro' ),
                                'hired'       => __( 'Hired', 'automatorwp-pro' ),
                                'rejected'    => __( 'Rejected', 'automatorwp-pro' ),
                                'archived'    => __( 'Archived', 'automatorwp-pro' ),
                            ),
                            'default' => 'interviewed'
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags( __( 'Job', 'automatorwp-pro' ) ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param string  $new_status New post status.
     * @param string  $old_status Old post status.
     * @param WP_Post $post       Post object.
     */
    public function listener( $new_status, $old_status, $post ) {

        // Bail if not is a job application
        if( $post->post_type !== 'job_application' ) {
            return;
        }

        // Bail if not status change
        if( $new_status === $old_status ) {
            return;
        }

        $job_id = $post->post_parent;
        $application_id = $post->ID;
        $user_id = get_post_meta( $application_id, '_candidate_user_id', true );
        $terms_ids = automatorwp_get_term_ids( $job_id, 'job_listing_type' );

        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user_id,
            'post_id'           => $job_id,
            'application_id'    => $application_id,
            'status'            => $new_status,
            'terms_ids'         => $terms_ids,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post, status and terms IDs are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['status'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if status doesn't matches with the trigger option
        if( $event['status'] !== $trigger_options['status'] ) {
            return false;
        }

        // Don't deserve if term doesn't matches with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_Job_Manager_Application_Status();